/*
 *  Demo.cpp - Demo sleketon
 *
 *  Written by Christian Bauer in 1998. Public domain.
 */

#include <AppKit.h>
#include <InterfaceKit.h>
#include <StorageKit.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include "Demo.h"


// Constants
const char APP_SIGNATURE[] = "application/x-vnd.cebix-Demos";


// Application object
class Demo : public BApplication {
public:
	Demo() : BApplication(APP_SIGNATURE) {}

	virtual void ReadyToRun(void)
	{
		// CD to application directory
		app_info the_info;
		GetAppInfo(&the_info);
		BEntry the_file(&the_info.ref);
		BEntry the_dir;
		the_file.GetParent(&the_dir);
		BPath the_path;
		the_dir.GetPath(&the_path);
		chdir(the_path.Path());

		// Open first window
		new DemoWindow();
	}

	virtual void AboutRequested(void)
	{
		BAlert *the_alert = new BAlert("", "Demo by Christian Bauer\n<cbauer@iphcip1.physik.uni-mainz.de>\nPublic domain.", "Neat");
		the_alert->Go();
	}
};


// Create application object and start it
int main(int argc, char **argv)
{
	srand(real_time_clock());
	Demo *the_app = new Demo();
	the_app->Run();
	delete the_app;
	return 0;
}


/*
 *  Window constructor
 */

DemoWindow::DemoWindow() : BDirectWindow(BRect(0, 0, DISPLAY_X-1, DISPLAY_Y-1), "Demo", B_TITLED_WINDOW, B_NOT_RESIZABLE | B_NOT_ZOOMABLE)
{
	// Move window to right position
	MoveTo(80, 60);

	// Keep the window aligned to 8-byte frame buffer boundaries
	SetWindowAlignment(B_BYTE_ALIGNMENT, 8);

	// Create transparent background view
	BView *back = new BView(Bounds(), "back", B_FOLLOW_NONE, B_WILL_DRAW);
	back->SetViewColor(B_TRANSPARENT_32_BIT);
	AddChild(back);

	// Switch to full-screen mode if window mode is not supported
	if (!SupportsWindowMode())
		SetFullScreen(true);

	// Create drawing semaphore
	drawing_sem = create_sem(0, "frame buffer access");

	// Start drawing thread
	drawing_thread = spawn_thread(thread_func, "redraw", B_DISPLAY_PRIORITY, this);
	resume_thread(drawing_thread);

	// Show the window
	Show();
	Sync();
}


/*
 *  Window destructor
 */

DemoWindow::~DemoWindow()
{
	// Quit drawing thread
	status_t l;
	acquire_sem(drawing_sem);
	delete_sem(drawing_sem);
	wait_for_thread(drawing_thread, &l);

	// Quit window
	Hide();
	Sync();
}


/*
 *  Window closed
 */

bool DemoWindow::QuitRequested(void)
{
	be_app->PostMessage(B_QUIT_REQUESTED);
	return true;
}


/*
 *  Window connected/disconnected
 */

void DemoWindow::DirectConnected(direct_buffer_info *info)
{
	switch (info->buffer_state & B_DIRECT_MODE_MASK) {
		case B_DIRECT_STOP:
			acquire_sem(drawing_sem);
			break;
		case B_DIRECT_MODIFY:
			acquire_sem(drawing_sem);
		case B_DIRECT_START:
			bits = (void *)((uint8 *)info->bits + info->window_bounds.top * info->bytes_per_row + info->window_bounds.left * info->bits_per_pixel / 8);
			bytes_per_row = info->bytes_per_row;
			pixel_format = info->pixel_format;
			unclipped = false;
			if (info->clip_list_count == 1)
				if (memcmp(&info->clip_bounds, &info->window_bounds, sizeof(clipping_rect)) == 0)
					unclipped = true;
			release_sem(drawing_sem);
			break;
	}
}


/*
 *  Drawing thread
 */

status_t DemoWindow::thread_func(void *arg)
{
	DemoWindow *obj = (DemoWindow *)arg;
	static int frames = 0;

	// Initialize effect
	obj->init_demo();
	bigtime_t start_time = system_time();

	// Do effect
	while (acquire_sem(obj->drawing_sem) == B_NO_ERROR) {
		bool drawn = false;
		if (obj->unclipped) {
			if (modifiers() & B_SCROLL_LOCK)
				drawn = false;
			else
				drawn = obj->redraw_demo(obj->bits, obj->bytes_per_row, obj->pixel_format);
		}
		release_sem(obj->drawing_sem);
		if (!drawn)
			snooze(20000);
		else
			frames++;
	}

	// Print frame rate
	bigtime_t end_time = system_time();
	printf("%f frames/sec\n", float(frames * 1000000) / float(end_time - start_time));

	// Deinitialize effect
	obj->exit_demo();
	return 0;
}


/*
 *  Mathematical functions
 */

float b_sqrt(float x) {
	uint32		val;
	float		y, z, t;
	float		flottant, tampon;
	
	flottant = x;
	val = *((uint32*)&flottant);
	val -= 0x3f800000L;
	val >>= 1;
	val += 0x3f800000L;
	val &= 0x7fffffffL;
	*((uint32*)&tampon) = val;
	y = tampon;
	z = y*y+x;
	t = y*y-x;
	y *= (float)4.0;
	x = z*z;
	t = t*t;
	y = z*y;
	t = (float)2.0*x-t;
	return  t/y;
}

float b_isqrt(float x) {
	uint32		val;
	float		y, z, t;
	float		flottant, tampon;
	
	flottant = x;
	val = *((uint32*)&flottant);
	val -= 0x3f800000L;
	val >>= 1;
	val += 0x3f800000L;
	val &= 0x7fffffffL;
	*((uint32*)&tampon) = val;
	y = tampon;
	z = y*y+x;
	t = y*y-x;
	y *= (float)4.0;
	x = z*z;
	t = t*t;
	y = z*y;
	t = (float)2.0*x-t;
	return  y/t;
}
