/*
 *  Plasma.cpp - Simple plasma fire
 *
 *  Written by Christian Bauer in 1998. Public domain.
 */

#include <math.h>
#include <stdlib.h>
#include <string.h>

#include "Demo.h"


// Global variables
uint8 PalRed[256];
uint8 PalGreen[256];
uint8 PalBlue[256];

uint8 ColorConv[256];

uint8 *buffer[2];
int draw_buffer = 1;


/*
 *  Initialize effect
 */

void DemoWindow::init_demo(void)
{
	int i;

	// Create color palette
	for (i=0; i<256; i++)
		PalRed[i] = PalGreen[i] = PalBlue[i] = 0;
	for (i=0; i<8; i++)		// 0..7: increase blue
		PalBlue[i] = i << 3;
	for (i=0; i<8; i++)		// 8..15: decrease blue
		PalBlue[i+8] = (7-i) << 3;
	for (i=8; i<32; i++)	// 8..31: increase red
		PalRed[i] = (i-8) * 255 / (32-8+1);
	for (i=32; i<56; i++) {	// 32..55: increase green
		PalRed[i] = 255;
		PalGreen[i] = (i-32) * 255 / (56-32+1);
	}
	for (i=56; i<80; i++) {	// 56..79: increase blue
		PalRed[i] = PalGreen[i] = 255;
		PalBlue[i] = (i-56) * 255 / (80-56+1);
	}
	for (i=80; i<256; i++)
		PalRed[i] = PalGreen[i] = PalBlue[i] = 255;

	// Create color conversion table
	{
		BScreen scr(this);
		for (int i=0; i<256; i++)
			ColorConv[i] = scr.IndexForColor(PalRed[i], PalGreen[i], PalBlue[i]);
	}

	// Allocate buffers
	buffer[0] = new uint8[DISPLAY_X * (DISPLAY_Y + 3)];
	buffer[1] = new uint8[DISPLAY_X * (DISPLAY_Y + 3)];
	memset(buffer[0], 0, DISPLAY_X * (DISPLAY_Y + 3));
	memset(buffer[1], 0, DISPLAY_X * (DISPLAY_Y + 3));
}


/*
 *  Main calculation
 */

bool DemoWindow::redraw_demo(void *bits, int32 bytes_per_row, color_space pixel_format)
{
	if (pixel_format != B_CMAP8)
		return false;

	uint8 *p, *q, *r;
	int x, y;
	int decr;

	// Copy up by 1 pixel and interpolate
	p = buffer[draw_buffer ^ 1] + 2 * DISPLAY_X; q = buffer[draw_buffer] - 1;
	decr = 1;
	for (y=0; y<DISPLAY_Y; y++) {
		for (x=0; x<DISPLAY_X; x++) {
			uint32 sum;
			sum = p[-DISPLAY_X-1];
			sum += p[-DISPLAY_X];
			sum += p[-DISPLAY_X+1];
			sum += p[-1];
			sum += p[1];
			sum += p[DISPLAY_X-1];
			sum += p[DISPLAY_X];
			sum += p[DISPLAY_X+1];
			sum >>= 3;
			if (sum)
				sum -= decr;	// Decrement every second row
			*++q = sum;
			p++;
		}
		decr ^= 1;
	}

	// Fill lower 3 rows (invisible) with random pixels
	p = buffer[draw_buffer] + DISPLAY_X * DISPLAY_Y + (DISPLAY_X-256) / 2 - 1;
	q = p + DISPLAY_X - 1;
	r = q + DISPLAY_X - 1;
	for (x=0; x<256; x++) {
		*++p = (rand() & 0xf) + 0x40;
		*++q = (rand() & 0xf) + 0x40;
		*++r = (rand() & 0xf) + 0x40;
	}

	// Insert random bright spots
	q = buffer[draw_buffer] + DISPLAY_X * (DISPLAY_Y+1) + (DISPLAY_X-256) / 2;
	int num_spots = rand() & 0xf;
	for (x=0; x<num_spots; x++) {
		p = q + (rand() & 0xff);
		p[-DISPLAY_X-1] = p[-DISPLAY_X] = p[-DISPLAY_X+1]
			= p[-1] = p[0] = p[1]
			= p[DISPLAY_X-1] = p[DISPLAY_X] = p[DISPLAY_X+1] = 255;
	}

	// Blit buffer to screen
	p = buffer[draw_buffer] - 1;
	q = (uint8 *)bits;
	for (y=0; y<DISPLAY_Y; y++) {
		for (x=0; x<DISPLAY_X/4; x++) {
#ifdef __INTEL__
			uint32 l = ColorConv[*++p];
			l |= ColorConv[*++p] << 8;
			l |= ColorConv[*++p] << 16;
			l |= ColorConv[*++p] << 24;
#else
			uint32 l = ColorConv[*++p] << 24;
			l |= ColorConv[*++p] << 16;
			l |= ColorConv[*++p] << 8;
			l |= ColorConv[*++p];
#endif
			((uint32 *)q)[x] = l;
		}
		q += bytes_per_row;
	}

	// Switch to other buffer
	draw_buffer ^= 1;
	return true;
}


/*
 *  Deinitialize effect
 */

void DemoWindow::exit_demo(void)
{
	// Delete buffers
	delete[] buffer[0];
	delete[] buffer[1];
}
